import { sequelize } from '../config/db';
import { QueryTypes } from 'sequelize';

export interface YogaTrainer {
  id: number;
  fullName: string;
  phone: string;
  email: string;
  qualification?: string;
  address?: string;
  state?: string;
  pincode?: string;
  expertise: string[];
  languages: string[];
  experience?: number;
  profileImage?: string;
  bio?: string;
  password: string;
  isVerified: boolean;
  createdAt: string;
  updatedAt: string;
}

export const createYogaTrainerAccount = async (
  fullName: string,
  phone: string,
  email: string,
  passwordHash: string,
  qualification: string,
  address: string,
  state: string,
  pincode: string,
  expertise: string[],
  languages: string[],
  experience: number,
  profileImage: string,
  bio: string
) => {
  try {
    await sequelize.query(
      `CALL CreateYogaTeacher(
        :fullName, :phone, :email, :password, :qualification,
        :address, :state, :pincode, :expertise, :languages,
        :experience, :profileImage, :bio
      )`,
      {
        replacements: {
          fullName,
          phone,
          email,
          password: passwordHash,
          qualification: qualification || null,
          address: address || null,
          state: state || null,
          pincode: pincode || null,
          expertise: JSON.stringify(expertise),
          languages: JSON.stringify(languages),
          experience: experience || null,
          profileImage: profileImage || null,
          bio: bio || null,
        },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    console.error('Error creating yoga trainer:', error);
    throw error;
  }
};

// FIND BY PHONE — 100% SAFE
export const findYogaTrainerByPhone = async (phone: string): Promise<YogaTrainer | null> => {
  try {
    const results: any = await sequelize.query(
      `SELECT * FROM yoga_teachers WHERE phone = :phone LIMIT 1`,
      {
        replacements: { phone },
        type: QueryTypes.SELECT,
      }
    );

    if (!results || results.length === 0) {
      return null;
    }

    const row = Array.isArray(results) ? results[0] : results;

    const safeJsonParse = (field: any): string[] => {
      if (!field) return [];
      try {
        return typeof field === 'string' ? JSON.parse(field) : [];
      } catch (e) {
        return [];
      }
    };

    return {
      id: row.id,
      fullName: row.fullName,
      phone: row.phone,
      email: row.email,
      password: row.password,
      qualification: row.qualification || '',
      address: row.address || '',
      state: row.state || '',
      pincode: row.pincode || '',
      expertise: safeJsonParse(row.expertise),
      languages: safeJsonParse(row.languages),
      experience: Number(row.experience) || 0,
      profileImage: row.profileImage || '',
      bio: row.bio || '',
      isVerified: Boolean(row.isVerified),
      createdAt: row.createdAt,
      updatedAt: row.updatedAt,
    };
  } catch (error) {
    console.error('Error in findYogaTrainerByPhone:', error);
    throw error;
  }
};

// GET ALL TRAINERS — 100% SAFE
export const getAllYogaTrainers = async (): Promise<YogaTrainer[]> => {
  try {
    const results: any = await sequelize.query(
      `SELECT * FROM yoga_teachers ORDER BY id DESC`,
      { type: QueryTypes.SELECT }
    );

    if (!results || results.length === 0) return [];

    const safeJsonParse = (field: any): string[] => {
      if (!field) return [];
      try {
        return typeof field === 'string' ? JSON.parse(field) : [];
      } catch {
        return [];
      }
    };

    return results.map((row: any) => ({
      id: row.id,
      fullName: row.fullName,
      phone: row.phone,
      email: row.email,
      qualification: row.qualification || '',
      address: row.address || '',
      state: row.state || '',
      pincode: row.pincode || '',
      expertise: safeJsonParse(row.expertise),
      languages: safeJsonParse(row.languages),
      experience: Number(row.experience) || 0,
      profileImage: row.profileImage || '',
      bio: row.bio || '',
      isVerified: Boolean(row.isVerified),
      createdAt: row.createdAt,
      updatedAt: row.updatedAt,
    }));
  } catch (error) {
    console.error('Error in getAllYogaTrainers:', error);
    throw error;
  }
};


// export const getYogaTeacherById = async (id: number): Promise<any | null> => {
//   try {
//     const result: any = await sequelize.query(
//       "CALL GetYogaTeacherById(:id)",
//       {
//         replacements: { id },
//         type: QueryTypes.SELECT,
//       }
//     );

//     if (!result || result.length === 0) return null;

//     const row = result[0];

//     return {
//       fullName: row.fullName || "Unknown Teacher",
//       expertise: row.expertise ? JSON.parse(row.expertise) : [],
//       languages: row.languages ? JSON.parse(row.languages) : [],
//       profileImage: row.profileImage || "/default-avatar.jpg",
//       bio: row.bio || "No bio available",
//     };
//   } catch (error) {
//     console.error("Error fetching yoga teacher:", error);
//     return null;
//   }
// };

export const getYogaTeacherById = async (id: number): Promise<any | null> => {
  try {
    const result: any = await sequelize.query(
      "CALL GetYogaTeacherById(:id)",
      { replacements: { id } }
    );
    const row = Array.isArray(result) ? result[0] : result;

    if (!row) return null;

    return {
      fullName: row.fullName ?? "Unknown Teacher",
      expertise: row.expertise ? JSON.parse(row.expertise) : [],
      languages: row.languages ? JSON.parse(row.languages) : [],
      profileImage: row.profileImage ?? "/default-avatar.jpg",
      bio: row.bio ?? "No bio available",
    };

  } catch (error) {
    console.error("Error fetching yoga teacher:", error);
    return null;
  }
};
