import { sequelize } from '../config/db';
import { QueryTypes } from 'sequelize';
import bcrypt from 'bcryptjs';



export interface User {
  id: number;
  name: string;
  contact: string;
  password: string;
}



// export const createUserAccount = async (name: string, Phone: string, passwordHash: string) => {
//   try {
//     await sequelize.query('CALL CreateUser(:name, :PhoneNumber, :password)', {
//       replacements: { name, PhoneNumber: Phone, password: passwordHash },
//       type: QueryTypes.RAW,
//     });

//   } catch (error) {
//     console.error('Error creating user:', error);
//     throw error;
//   }
// };


export const createUserAccount = async (
  name: string,
  Phone: string,
  passwordHash: string,
  email: string,
  address: string,
  state: string,
  pincode: string,
  dob: string,
  birthplace: string
) => {
  try {
    await sequelize.query(
      'CALL CreateUser(:name, :phoneNumber, :password, :email, :address, :state, :pincode, :dob, :birthplace)',
      {
        replacements: {
          name,
          phoneNumber: Phone,     // ✔ must match :phoneNumber
          password: passwordHash,
          email,
          address,
          state,
          pincode,
          dob,
          birthplace
        },
        type: QueryTypes.RAW,
      }
    );

  } catch (error) {
    console.error('Error creating user:', error);
    throw error;
  }
};


 export const findUserByContact = async (contact: string): Promise<User | null> => {
  try {

    const result = await sequelize.query('CALL GetUserWithContact(:contact)', {
      replacements: { contact },
      type: QueryTypes.SELECT,
    });
    
    const firstResult = result[0] as Record<string, User> | undefined;

    if (!firstResult) return null;

    const user = firstResult[Object.keys(firstResult)[0]];

    return user ?? null;
  } catch (error) {
    console.error('Error fetching user by contact:', error);
    throw error;
  }
};



export const updateUserPassword = async (contact: string, hashedPassword: string): Promise<void> => {
  try {
    await sequelize.query(
      'CALL UpdateUserPassword(:contact, :password)',
      {
        replacements: { contact, password: hashedPassword },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    console.error('Error updating user password:', error);
    throw error;
  }
};


export const updateUserPass = async (
  user_id: number,
  current_password: string,
  new_password: string
): Promise<void> => {
  try {
    
    const [user]: any = await sequelize.query(
      'SELECT password FROM users_table WHERE id = :user_id',
      {
        replacements: { user_id },
        type: QueryTypes.SELECT,
      }
    );

    if (!user || !user.password) {
      throw new Error('User not found');
    }

    
    const isMatch = await bcrypt.compare(current_password, user.password);
    if (!isMatch) {
      throw new Error('Incorrect current password');
    }

 
    const hashedPassword = await bcrypt.hash(new_password, 10);


    await sequelize.query(
      'CALL UpdateUserPass(:user_id, :hashedPassword)',
      {
        replacements: { user_id, hashedPassword },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    throw error;
  }
};

export const getAllUsers = async () => {
  try {
    const result = await sequelize.query(
      "CALL GetAllUsers()",
      {
        type: QueryTypes.SELECT
      }
    );

    return result; // array of all users
  } catch (error) {
    console.error("Error fetching all users:", error);
    throw error;
  }
};
