import { sequelize } from '../config/db';
import { QueryTypes } from 'sequelize';

export interface Astrologer {
  id: number;
  username: string;
  email: string | null;
  address: string;
  gender: string;
  mobileNumber: string;
  pincode: string;
  state: string;
  specialization: string;
  languages: string;
  experience: number;
  education: string;
  description: string;
  profileImage: string;
  price: number;
  is_verified: number;
  last_active: string;
  average_rating: number;
  number_of_reviews: number;
  rating_score: string;

}

export interface User {
  id: number;
  name: string;
  contact: string;
  gender: string;
  email: string;
  // wallet :string;
}

type WalletResult = {
  astrologer_earnings: number;
  total_withdrawals: number;
  remaining_balance: number;
};

type Conversation = {
  date: string;
  total_conversations: number;
  total_chat_duration_minutes: number;
};

type DailyEarning = {
  date: string;
  daily_earning: number;
  astrologer_earning: number;
};


export const fetchAllAstrologers = async (): Promise<Astrologer[] | null> => {
    try {
      
      const results = await sequelize.query('CALL GetVerifiedAstrologers()', {
        type: QueryTypes.RAW,
      });
  
      
      const astrologers: Astrologer[] = Array.isArray(results) ? (results as Astrologer[]) : [];
  
      return astrologers || null;
    } catch (error) {
      console.error('Error fetching astrologers:', error);
      throw new Error('Failed to fetch astrologer data');
    }
  };

  export const fetchastrologerdata = async (contact: string): Promise<Astrologer | null> => {
    try {
  
      const result = await sequelize.query('CALL GetAstrologerdataWithContact(:contact)', {
        replacements: { contact },               
        type: QueryTypes.RAW,
      });
  
      const astrologer: Astrologer[] = Array.isArray(result) ? (result as Astrologer[]) : [];
  
      return astrologer[0] || null;
    } catch (error) {
      console.error('Error fetching user by contact:', error);
      throw error;
    }
  };


  export const fetchuserdata = async (contact: string): Promise<User | null> => {
    try {
  
      const result = await sequelize.query('CALL GetUserdataWithContact(:contact)', {
        replacements: { contact },
        type: QueryTypes.RAW,
      });
  
      const users: User[] = Array.isArray(result) ? (result as User[]) : [];
  
      return users[0] || null;
    } catch (error) {
      console.error('Error fetching user by contact:', error);
      throw error;
    }
  };
  
//   export const updateUserDetails = async (contact: string, name: string, email: string, gender: string): Promise<User | null> => {
//     try {
//         const result = await sequelize.query('CALL UpdateUser(:contact, :name, :email, :gender)', {
//             replacements: { contact, name, email, gender },
//             type: QueryTypes.RAW,
//         });

//         const updatedUser: User[] = Array.isArray(result) ? (result as User[]) : [];

//         return updatedUser[0] || null;
//     } catch (error) {
//         console.error('Error updating user details:', error);
//         throw new Error('Failed to update user data');
//     }
// };
  

// export const updateUserDetails = async (
//   contact: string,
//   name: string,
//   email: string,
//   gender: string
// ): Promise<boolean> => {
//   try {
//     const result = await sequelize.query(
//       `
//       UPDATE users_table
//       SET 
//         username = :name,
//         email = :email,
//         gender = :gender,
//         updated_at = NOW()
//       WHERE mobileNumber = :contact
//       `,
//       {
//         replacements: { name, email, gender, contact },
//         type: QueryTypes.UPDATE,
//       }
//     );

//     // MySQL returns affected rows count at index 1
//     const affectedRows = result[1] as number;

//     return affectedRows > 0;
//   } catch (error) {
//     console.error("Error updating user details:", error);
//     throw new Error("Failed to update user data");
//   }
// };

export const updateUserDetails = async (
  contact: string,
  name: string,
  email: string,
  gender: string
): Promise<boolean> => {
  try {
    const result: any = await sequelize.query(
      "CALL UpdateUser(:contact, :name, :email, :gender)",
      {
        replacements: {
          contact,
          name,
          email,
          gender,
        },
      }
    );

    return true; 
  } catch (error) {
    console.error("Error updating user details:", error);
    throw new Error("Failed to update user data");
  }
};


export const fetchwalletbalance= async(userId: number)=>{

  try {
    const result = await sequelize.query('CALL Getuserwallet(:userId)', {
        replacements: { userId },
        type: QueryTypes.RAW,
    });

    return result[0]|| null;
} catch (error) {
    console.error('Error getting user wallet', error);
    throw new Error('Failed to fetch user wallet');
}
}
 


export const fetchastrologerwalletdata = async (contact: string) => {
  try {
   
    const [walletSummary] = await sequelize.query<WalletResult>(
      `
      SELECT 
        astrologer_share AS astrologer_earnings,
        0 AS total_withdrawals,
        wallet_balance AS remaining_balance
      FROM astrologer_wallet
      WHERE mobileNumber = :contact
      `,
      {
        replacements: { contact },
        type: QueryTypes.SELECT
      }
    );

  
    const dailyConversations = await sequelize.query<Conversation>(
      `
      SELECT 
        DATE(created_at) AS date,
        COUNT(*) AS total_conversations,
        ROUND(SUM(total_time) / 60, 2) AS total_chat_duration_minutes
      FROM astrologer_user_messages
      WHERE astrologer_number = :contact
      GROUP BY DATE(created_at)
      ORDER BY DATE(created_at) DESC
      `,
      {
        replacements: { contact },
        type: QueryTypes.SELECT
      }
    );

    
    const dailyEarnings = await sequelize.query<DailyEarning>(
      `
      SELECT 
        DATE(created_at) AS date,
        SUM(amount) AS daily_earning,
        SUM(amount * (1 - commission_percentage / 100)) AS astrologer_earning
      FROM astrologer_user_messages
      WHERE astrologer_number = :contact
      GROUP BY DATE(created_at)
      ORDER BY DATE(created_at) DESC
      `,
      {
        replacements: { contact },
        type: QueryTypes.SELECT
      }
    );

    return {
      astrologer_earnings: Number(walletSummary?.astrologer_earnings || 0),
      total_withdrawals: Number(walletSummary?.total_withdrawals || 0),
      remaining_balance: Number(walletSummary?.remaining_balance || 0),
      daily_conversations: dailyConversations,
      daily_earnings: dailyEarnings
    };

  } catch (error) {
    console.error('Error getting astrologer wallet:', error);
    throw new Error('Failed to fetch astrologer wallet');
  }
};


export const UpdateAstroProfileImage =async ( Profile_image: string, contact:string)=>{
  
  try{
    await sequelize.query(
      'CALL UpdateAstroProfileImage(:Profile_image, :contact)',
      {
        replacements: { Profile_image,  contact },
        type: QueryTypes.RAW,
      }
    );
  }catch(error){
    console.log("Error find in procedure.");
    throw new Error('Failed to update image.');
  }
}


// model for pooja


export const fetchAllPoojas = async () => {
  try {
    const result = await sequelize.query('CALL GetallPooja()', {
      type: QueryTypes.RAW,  
    });

    return result || null;
  } catch (error) {
    console.error("Error in fetchAllPoojas:", error);
    throw new Error('Failed to fetch pooja.');
  }
};


// fetch pooja details
export const fetchpoojadetails= async(Poojaid: number)=>{

  try {
    const result = await sequelize.query('CALL GetPoojaDetails(:Poojaid)', {
        replacements: { Poojaid },
        type: QueryTypes.RAW,
    });

    return result[0]|| null;
} catch (error) {
    console.error('Error getting user wallet', error);
    throw new Error('Failed to fetch user wallet');
}
}



export const fetchmarquee= async()=>{

  try {
    const result = await sequelize.query('CALL GetLatestMarqueeBanner()', {
        type: QueryTypes.RAW,
    });

    return result || null;
} catch (error) {
    console.error('Error getting marquee text.', error);
    throw new Error('Failed to fetch maruee deatls');
}
}



export const fetchastrobank= async(astro_id: number)=>{

  try {
    const result = await sequelize.query('CALL GetAstroBank(:astro_id)', {
        replacements: { astro_id },
        type: QueryTypes.RAW,
    });

    return result[0] || null;
} catch (error) {
    console.error('Error getting astro bank', error);
    throw new Error('Failed to fetch astrologer bank.');
}
}

export const updateAstroBank = async (
  astro_id: number,
  account_holder: string,
  bank_name: string,
  account_number: string,
  ifsc_code: string
) => {
  try {
    await sequelize.query('CALL UpdateAstroBank(:astro_id, :account_holder, :bank_name, :account_number, :ifsc_code)', {
      replacements: {
        astro_id,
        account_holder,
        bank_name,
        account_number,
        ifsc_code
      },
      type: QueryTypes.RAW,
    });
    return true;
  } catch (error) {
    console.error('Error updating astro bank', error);
    throw new Error('Failed to update astrologer bank.');
  }
};


export const fetchusechathistory= async(Phone: string)=>{

  try {
    const result = await sequelize.query('CALL GetTop8UserChats(:Phone)', {
        replacements: { Phone },
        type: QueryTypes.RAW,
    });

    return result || null;
} catch (error) {
    console.error('Error getting user chats', error);
    throw new Error('Failed to fetch User Chat history.');
}
}

export const fetchAstrologerInteractionHistory = async (
  astrologerId: number,
  fromDate?: string,
  toDate?: string,
  limit: number = 10,
  offset: number = 0
) => {
  try {
    // Stored procedure with pagination
    const data = await sequelize.query(
      'CALL GetAstrologerInteractions(:astrologerId, :fromDate, :toDate, :limit, :offset)',
      {
        replacements: { astrologerId, fromDate: fromDate || null, toDate: toDate || null, limit, offset },
        // type: QueryTypes.SELECT,
      }
    );

    // return data[0] || []; 
    return Array.isArray(data) ? data : [];
  } catch (error) {
    console.error('Error fetching astrologer interaction history:', error);
    throw new Error('Failed to fetch astrologer interaction history');
  }
};

// ADMIN – fetch astrologer wallet balance list

export const fetchAdminAstrologerWallets = async () => {
  try {
    const result = await sequelize.query(
      `
      SELECT 
        a.id AS astrologer_id,
        a.username,
        a.mobileNumber,
        IFNULL(w.total_earned, 0) AS total_earned,
        IFNULL(w.wallet_balance, 0) AS wallet_balance
      FROM astrologer a
      LEFT JOIN astrologer_wallet w 
        ON w.astrologer_id = a.id
      ORDER BY w.wallet_balance DESC
      `,
      {
        type: QueryTypes.SELECT,
      }
    );

    return result || [];
  } catch (error) {
    console.error('Error fetching admin astrologer wallets', error);
    throw new Error('Failed to fetch astrologer wallet data');
  }
};
