import { sequelize } from "../config/db";
import { QueryTypes } from "sequelize";

interface PayoutPayload {
  astrologer_id: number;
  amount: number;
  payment_mode: "Bank" | "UPI";
  reference_id: string;   // 👈 ADMIN INPUT
  remarks?: string;
}

export const payoutAstrologerAmount = async ({
  astrologer_id,
  amount,
  payment_mode,
  reference_id,
  remarks,
}: PayoutPayload) => {
  try {
    // 1️⃣ Wallet fetch
    const wallet: any[] = await sequelize.query(
      `
      SELECT wallet_balance 
      FROM astrologer_wallet 
      WHERE astrologer_id = :id
      `,
      {
        replacements: { id: astrologer_id },
        type: QueryTypes.SELECT,
      }
    );

    if (!wallet.length) {
      throw new Error("Wallet not found");
    }

    const beforeBalance = Number(wallet[0].wallet_balance);

    if (amount > beforeBalance) {
      throw new Error("Insufficient wallet balance");
    }

    const afterBalance = beforeBalance - amount;

    // 2️⃣ Update wallet balance
    await sequelize.query(
      `
      UPDATE astrologer_wallet
      SET wallet_balance = :after
      WHERE astrologer_id = :id
      `,
      {
        replacements: { after: afterBalance, id: astrologer_id },
        type: QueryTypes.UPDATE,
      }
    );

    // 3️⃣ Insert payout history (reference ADMIN se)
    await sequelize.query(
      `
      INSERT INTO astrologer_payout_history
      (
        astrologer_id,
        paid_amount,
        before_balance,
        after_balance,
        payment_mode,
        reference_id,
        remarks
      )
      VALUES
      (
        :id,
        :paid,
        :before,
        :after,
        :mode,
        :ref,
        :remarks
      )
      `,
      {
        replacements: {
          id: astrologer_id,
          paid: amount,
          before: beforeBalance,
          after: afterBalance,
          mode: payment_mode,
          ref: reference_id,   // 👈 direct use
          remarks: remarks || null,
        },
        type: QueryTypes.INSERT,
      }
    );

    return {
      paid_amount: amount,
      remaining_balance: afterBalance,
      reference_id,
    };
  } catch (error) {
    console.error("Error in payoutAstrologerAmount:", error);
    throw error;
  }
};



export const fetchAstrologerPayoutHistory = async (
  astrologer_id: number
) => {
  try {
    const result = await sequelize.query(
      `
      SELECT
        p.id,
        p.paid_amount,
        p.before_balance,
        p.after_balance,
        p.payment_mode,
        p.reference_id,
        p.remarks,
        p.created_at,
        a.username AS astrologer_name,
        a.mobileNumber
      FROM astrologer_payout_history p
      JOIN astrologer a ON a.id = p.astrologer_id
      WHERE p.astrologer_id = :id
      ORDER BY p.created_at DESC
      `,
      {
        replacements: { id: astrologer_id },
        type: QueryTypes.SELECT,
      }
    );

    return result || [];
  } catch (error) {
    console.error("Error fetching payout history:", error);
    throw new Error("Failed to fetch payout history");
  }
};




/**
 * 🔹 Total Admin Earning (All Time)
 */


export const getTotalAdminEarning = async () => {
  try {
    const result: any[] = await sequelize.query(
      `
      SELECT 
        SUM(aum.total_amount * a.commission_percentage / 100)
          AS total_admin_earning
      FROM astrologer_user_messages aum
      JOIN astrologer a 
        ON a.mobileNumber = aum.astrologer_number
      WHERE aum.status = 'end'
      `,
      { type: QueryTypes.SELECT }
    );

    return result[0]?.total_admin_earning || 0;
  } catch (error) {
    console.error("Error in getTotalAdminEarning:", error);
    throw error;
  }
};

/**
 * 🔹 Today Admin Earning
 */
export const getTodayAdminEarning = async () => {
  const result: any[] = await sequelize.query(
    `
    SELECT 
      SUM(aum.total_amount * a.commission_percentage / 100)
        AS today_admin_earning
    FROM astrologer_user_messages aum
    JOIN astrologer a 
      ON a.mobileNumber = aum.astrologer_number
    WHERE aum.status = 'end'
    AND DATE(aum.created_at) = CURDATE()
    `,
    { type: QueryTypes.SELECT }
  );

  return result[0]?.today_admin_earning || 0;
};


/**
 * 🔹 Date-wise Admin Earning (for charts)
 */
export const getDateWiseAdminEarning = async () => {
  return await sequelize.query(
    `
    SELECT
      DATE(aum.created_at) AS earning_date,
      SUM(aum.total_amount * a.commission_percentage / 100)
        AS admin_earning
    FROM astrologer_user_messages aum
    JOIN astrologer a 
      ON a.mobileNumber = aum.astrologer_number
    WHERE aum.status = 'end'
    GROUP BY DATE(aum.created_at)
    ORDER BY earning_date DESC
    `,
    { type: QueryTypes.SELECT }
  );
};

/**
 * 🔹 Admin Earning by Specific Date
 */
export const getAdminEarningByDate = async (date: string) => {
  try {
    const result: any[] = await sequelize.query(
      `
      SELECT
        SUM(total_amount * commission_percentage / 100) AS admin_earning
      FROM astrologer_user_messages
      WHERE status = 'end'
      AND DATE(created_at) = :date
      `,
      {
        replacements: { date },
        type: QueryTypes.SELECT,
      }
    );

    return result[0]?.admin_earning || 0;
  } catch (error) {
    console.error("Error in getAdminEarningByDate:", error);
    throw error;
  }
};





export const getTodayAstrologerWiseAdminEarning = async () => {
  try {
    const result = await sequelize.query(
      `
      SELECT
        a.id AS astrologer_id,
        a.username AS astrologer_name,
        a.mobileNumber,
        COUNT(aum.id) AS total_calls,
        SUM(aum.total_amount) AS total_collection,
        a.commission_percentage,
        SUM(aum.total_amount * a.commission_percentage / 100)
          AS admin_earning
      FROM astrologer_user_messages aum
      JOIN astrologer a 
        ON a.mobileNumber = aum.astrologer_number
      WHERE aum.status = 'end'
      AND DATE(aum.created_at) = CURDATE()
      GROUP BY a.id, DATE(aum.created_at)
      ORDER BY admin_earning DESC
      `,
      {
        type: QueryTypes.SELECT,
      }
    );

    return result || [];
  } catch (error) {
    console.error("Error in getTodayAstrologerWiseAdminEarning:", error);
    throw error;
  }
};

export const getDateAstrologerWiseAdminEarning = async (date: string) => {
  try {
    const result = await sequelize.query(
      `
      SELECT
        a.id AS astrologer_id,
        a.username AS astrologer_name,
        a.mobileNumber,
        COUNT(aum.id) AS total_calls,
        SUM(aum.total_amount) AS total_collection,
        a.commission_percentage,
        SUM(aum.total_amount * a.commission_percentage / 100)
          AS admin_earning
      FROM astrologer_user_messages aum
      JOIN astrologer a 
        ON a.mobileNumber = aum.astrologer_number
      WHERE aum.status = 'end'
      AND DATE(aum.created_at) = :date
      GROUP BY a.id, DATE(aum.created_at)
      ORDER BY admin_earning DESC
      `,
      {
        replacements: { date },
        type: QueryTypes.SELECT,
      }
    );

    return result || [];
  } catch (error) {
    console.error("Error in getDateAstrologerWiseAdminEarning:", error);
    throw error;
  }
};
