import { sequelize } from "../config/db";
import { QueryTypes } from "sequelize";

export interface Pandit {
  id: number;
  username: string;
  email: string;
  password: string;
  address: string;
  gender: "Male" | "Female" | "Other";
  mobileNumber: string;
  pincode: string;
  state: string;
  PoojaSpecialist: string;
  languages: string;
  experience: string;
  education: string;
  description: string;
  profileImage: string;
  is_verified: number;
  created_at: Date;
}

export const createPanditAccount = async (
  username: string,
  email: string,
  passwordHash: string,
  address: string,
  gender: "Male" | "Female" | "Other",
  mobileNumber: string,
  pincode: string,
  state: string,
  PoojaSpecialist: string,
  languages: string,
  experience: string,
  education: string,
  description: string,
  profileImage: string,
) => {
  try {
    await sequelize.query(
      `CALL CreatePandit(
        :username,
        :email,
        :password,
        :address,
        :gender,
        :mobileNumber,
        :pincode,
        :state,
        :PoojaSpecialist,
        :languages,
        :experience,
        :education,
        :description,
        :profileImage
      )`,
      {
        replacements: {
          username,
          email,
          password: passwordHash,
          address,
          gender,
          mobileNumber,
          pincode,
          state,
          PoojaSpecialist,
          languages,
          experience,
          education,
          description,
          profileImage,
        },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    // console.error("Error creating pandit:", error);
    throw error;
  }
};

export const findPanditByContact = async (contact: string): Promise<Pandit | null> => {
  try {
    const result: any = await sequelize.query(
      "CALL GetPanditWithContact(:contact)",
      { replacements: { contact }, type: QueryTypes.SELECT }
    );

    if (!result || result.length === 0) return null;

    return result[0] as Pandit;
  } catch (error) {
    // console.error("Error fetching pandit by contact:", error);
    throw error;
  }
};

export const getAllPandits = async (): Promise<Pandit[]> => {
  try {
    const result: any = await sequelize.query("CALL GetAllPandits()", {
      type: QueryTypes.SELECT,
    });

    // console.log("Raw result from GetAllPandits:", JSON.stringify(result, null, 2));

    // Extract pandits from nested structure and filter out metadata
    let data: any[] = Array.isArray(result) ? result : [result];
    // If first element is an object with numbered keys, extract values
    if (data[0] && typeof data[0] === "object" && !Array.isArray(data[0])) {
      data = Object.values(data[0]);
    }

    const pandits: Pandit[] = data
      .filter(
        (item) =>
          item &&
          typeof item === "object" &&
          "id" in item &&
          !("fieldCount" in item)
      )
      .map((item) => ({
        id: Number(item.id),
        username: item.username || "Unknown",
        email: item.email || "N/A",
        password: item.password || "",
        address: item.address || "N/A",
        gender: item.gender || "Other",
        mobileNumber: item.mobileNumber || item.phone || "N/A",
        pincode: item.pincode || "N/A",
        state: item.state || "N/A",
        PoojaSpecialist: item.PoojaSpecialist || item.pooja_specialist || "N/A",
        languages: item.languages || "N/A",
        experience: item.experience || "N/A",
        education: item.education || "N/A",
        description: item.description || "N/A",
        profileImage: item.profileImage || item.profile_image || "",
        is_verified: Number(item.is_verified || 0),
        created_at: item.created_at ? new Date(item.created_at) : new Date(),
        // identityProofType: item.identityProofType || item.identity_proof_type || "OTHER",
        // identityProof: item.identityProof || item.identity_proof || "",
      }));

    // console.log("Processed pandits in model:", JSON.stringify(pandits, null, 2));
    return pandits;
  } catch (error) {
    // console.error("Error fetching all pandits:", error);
    throw error;
  }
};

export const getPanditById = async (id: number): Promise<Pandit | null> => {
  try {
    const result: any = await sequelize.query(
      "CALL GetPanditById(:id)",
      {
        replacements: { id },
        type: QueryTypes.SELECT,
      }
    );

    if (!result || result.length === 0) return null;

    return result[0] as Pandit;
  } catch (error) {
    throw error;
  }
};



export const verifyPanditById = async (id: number): Promise<{ affectedRows: number }> => {
  try {
    const [rows]: any = await sequelize.query("CALL VerifyPanditById(:id)", {
      replacements: { id },
    });

    // CALL ka response array hota hai → [{ affectedRows: 1 }]
    const affectedRows = rows?.[0]?.affectedRows || 0;
    return { affectedRows };
  } catch (error) {
    console.error("Error verifying pandit:", error);
    throw error;
  }
};


export const deletePanditById = async (id: number): Promise<{ affectedRows: number }> => {
  try {
    const [result]: any = await sequelize.query(
      "DELETE FROM `Pandit` WHERE id = :id",  // ✅ correct table name
      {
        replacements: { id },
        type: QueryTypes.DELETE,
      }
    );

    return { affectedRows: result?.affectedRows || result?.rowCount || 0 };
  } catch (error) {
    console.error("Error deleting pandit:", error);
    throw error;
  }
};


// Unverify pandit using stored procedure
export const unverifyPanditById = async (id: number): Promise<{ affectedRows: number }> => {
  try {
    const [rows]: any = await sequelize.query("CALL UnverifyPanditById(:id)", {
      replacements: { id },
    });

    // CALL ka response array hota hai → [{ affectedRows: 1 }]
    const affectedRows = rows?.[0]?.affectedRows || 0;
    return { affectedRows };
  } catch (error) {
    console.error("Error unverifying pandit:", error);
    throw error;
  }
};

