import { sequelize } from '../config/db';
import { QueryTypes } from 'sequelize';
import bcrypt from 'bcryptjs';

export interface Astrologer {
  id: number;
  Astrologer_name: string;
  Phone: string;
  Email: string;
  Address: string;
  Pincode: string;
  State: string;
  Specialization: string[];
  Languages: string[];
  Experience: string;
  Highest_qualification: string;
  Description: string;
  Profile_image: string;
  password: string;
}





export const createAstrologerAccount = async (
  Astrologer_name: string,
  Phone: string,
  Email: string,
  Address: string,
  Pincode: string,
  State: string,
  Specialization: string[],
  Languages: string[],
  Experience: string,
  Highest_qualification: string,
  Description: string,
  Profile_image: string,
  passwordHash: string
) => {
  try {
    await sequelize.query(
      `CALL CreateAstrologer(
        :Astrologer_name,
        :Phone,
        :Email,
        :Address,
        :Pincode,
        :State,
        :Specialization,
        :Languages,
        :Experience,
        :Highest_qualification,
        :Description,
        :Profile_image,
        :Password
      )`,
      {
        replacements: {
          Astrologer_name,
          Phone,
          Email,
          Address,
          Pincode,
          State,
          Specialization: JSON.stringify(Specialization),
          Languages: JSON.stringify(Languages),
          Experience,
          Highest_qualification,
          Description,
          Profile_image,
          Password: passwordHash,
        },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    console.error('Error creating astrologer:', error);
    throw error;
  }
};


export const findAstrologerByContact = async (contact: string): Promise<Astrologer | null> => {
  try {
    
    const result = await sequelize.query('CALL GetAstrologerWithContact(:contact)', {
      replacements: { contact },
      type: QueryTypes.SELECT,
    }) ;

  

    const firstResult = result[0] as Record<string, Astrologer> | undefined;

    if (!firstResult) return null;

    const astrologer = firstResult[Object.keys(firstResult)[0]];
    

    return astrologer ?? null;
  } catch (error) {
    console.error('Error fetching astrologer by contact:', error);
    throw error;
  }
};



export const updateAstrologerPassword = async (contact: string, hashedPassword: string): Promise<void> => {
  try {
    await sequelize.query(
      'CALL UpdateAstrologerPassword(:contact, :password)',
      {
        replacements: { contact, password: hashedPassword },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    console.error('Error updating astrologer password:', error);
    throw error;
  }
};



export const updateAstrorPass = async (
  astro_id: number,
  current_password: string,
  new_password: string
): Promise<void> => {
  try {
    
    const [astro]: any = await sequelize.query(
      'SELECT password FROM astrologer WHERE id = :astro_id',
      {
        replacements: { astro_id },
        type: QueryTypes.SELECT,
      }
    );

    if (!astro || !astro.password) {
      throw new Error('Astrologer not found');
    }

    // Step 2: Compare current_password with DB password
    const isMatch = await bcrypt.compare(current_password, astro.password);
    if (!isMatch) {
      throw new Error('Incorrect current password');
    }

    // Step 3: Hash new password
    const hashedPassword = await bcrypt.hash(new_password, 10);

    // Step 4: Call procedure to update password
    await sequelize.query(
      'CALL UpdateAstroPassword(:astro_id, :hashedPassword)',
      {
        replacements: { astro_id, hashedPassword },
        type: QueryTypes.RAW,
      }
    );
  } catch (error) {
    throw error;
  }
};



export const verifyAstrologer = async (astro_id: number): Promise<void> => {
  try {
    await sequelize.query(
      "UPDATE astrologer SET is_verified = 1 WHERE id = :astro_id",
      {
        replacements: { astro_id },
        type: QueryTypes.UPDATE,
      }
    );
  } catch (error) {
    console.error("Error verifying astrologer:", error);
    throw error;
  }
};

// ✅ Unverify astrologer
export const unverifyAstrologer = async (astro_id: number): Promise<void> => {
  try {
    await sequelize.query(
      "UPDATE astrologer SET is_verified = 0 WHERE id = :astro_id",
      {
        replacements: { astro_id },
        type: QueryTypes.UPDATE,
      }
    );
  } catch (error) {
    console.error("Error unverifying astrologer:", error);
    throw error;
  }
};








// 🔹 Update price & commission
export const updateAstrologerPriceAndCommission = async (
  astro_id: number,
  price: number,
  commission_percentage: number
): Promise<void> => {
  try {
    await sequelize.query(
      `UPDATE astrologer 
       SET price = :price, commission_percentage = :commission_percentage 
       WHERE id = :astro_id`,
      {
        replacements: { astro_id, price, commission_percentage },
        type: QueryTypes.UPDATE,
      }
    );
  } catch (error) {
    console.error("Error updating astrologer price/commission:", error);
    throw error;
  }
};

// 🔹 Get existing price & commission
export const getAstrologerPriceAndCommission = async (
  astro_id: number
): Promise<{ price: number; commission_percentage: number } | null> => {
  try {
    const [result]: any = await sequelize.query(
      `SELECT price, commission_percentage 
       FROM astrologer 
       WHERE id = :astro_id`,
      {
        replacements: { astro_id },
        type: QueryTypes.SELECT,
      }
    );

    if (!result) return null;
    return result;
  } catch (error) {
    console.error("Error fetching astrologer price/commission:", error);
    throw error;
  }
};


export const fetchAllAstrologers = async (): Promise<Astrologer[]> => {
  try {
    const results: any = await sequelize.query(
      `SELECT 
         id,
         username,
         mobileNumber,
         email,
         address,
         pincode,
         state,
         specialization,
         languages,
         experience,
         education,
         description,
         profileImage,
         password,
         price,
         commission_percentage,
         is_verified
       FROM astrologer`,
      { type: QueryTypes.SELECT }
    );

    return results.map((row: any) => {
      let specialization = [];
      let languages = [];

      try {
        specialization = row.specialization
          ? JSON.parse(row.specialization)
          : [];
      } catch {
        specialization = row.specialization
          ? row.specialization.split(",").map((s: string) => s.trim())
          : [];
      }

      try {
        languages = row.languages ? JSON.parse(row.languages) : [];
      } catch {
        languages = row.languages
          ? row.languages.split(",").map((l: string) => l.trim())
          : [];
      }

      return {
        ...row,
        specialization,
        languages,
      };
    });
  } catch (error) {
    console.error("Error fetching astrologers:", error);
    throw error;
  }
};



export const toggleAstrologerFreeStatusByMobile = async (mobile: string): Promise<any> => {
  try {
    const result: any = await sequelize.query(
      `CALL ToggleAstrologerFreeStatusByMobile(?)`,  
      {
        replacements: [mobile],  
        type: QueryTypes.RAW,
      }
    );

    let row = null;

    if (result && result[0]) {
      if (Array.isArray(result[0])) {
        row = result[0][0];
      } else if (result[0][0]) {
        row = result[0][0];
      } else if (typeof result[0] === 'object') {
        row = result[0];
      }
    }

    if (!row && result && result[1]) {
      if (Array.isArray(result[1])) {
        row = result[1][0];
      } else if (result[1][0]) {
        row = result[1][0];
      }
    }

    if (!row) {
      throw new Error("No data returned from stored procedure. Check console log.");
    }

    return row;

  } catch (error: any) {
    console.error("Error in toggleAstrologerFreeStatusByMobile:", error);
    throw error;
  }
};

export const fetchAllAstrologersWithPriceCommission = async (): Promise<any[]> => {
  try {
    const results: any = await sequelize.query(
      `SELECT 
         id,
         username,
         mobileNumber,
         email,
         address,
         pincode,
         state,
         specialization,
         languages,
         experience,
         education,
         description,
         profileImage,
         price,
         commission_percentage,
         is_verified,
         is_free,
         original_price,
         created_at
       FROM astrologer 
       ORDER BY id DESC`,
      { type: QueryTypes.SELECT }
    );

    return results.map((row: any) => {
      let specialization = [];
      let languages = [];

      try {
        specialization = row.specialization ? JSON.parse(row.specialization) : [];
      } catch {
        specialization = row.specialization ? row.specialization.split(",").map((s: string) => s.trim()) : [];
      }

      try {
        languages = row.languages ? JSON.parse(row.languages) : [];
      } catch {
        languages = row.languages ? row.languages.split(",").map((l: string) => l.trim()) : [];
      }

      return {
        ...row,
        specialization,
        languages,
        price: row.price ?? null,
        commission_percentage: row.commission_percentage ?? null,
        is_free: row.is_free ?? 0,
        original_price: row.original_price ?? null,
      };
    });
  } catch (error) {
    console.error("Error fetching astrologers with price/commission:", error);
    throw error;
  }
};