#!/bin/bash

# AWS Lambda Cron Job Deployment Script
# This script deploys the panchang cron job Lambda function

set -e

echo "🚀 Starting Lambda deployment..."

# Check if AWS CLI is installed
if ! command -v aws &> /dev/null; then
    echo "❌ AWS CLI is not installed. Please install it first."
    exit 1
fi

# Check if AWS credentials are configured
if ! aws sts get-caller-identity &> /dev/null; then
    echo "❌ AWS credentials not configured. Please run 'aws configure' first."
    exit 1
fi

# Variables
STACK_NAME="panchang-cron-stack"
REGION="us-east-1"  # Change this to your preferred region

echo "📦 Installing dependencies..."
npm install

echo "🗜️  Creating deployment package..."
zip -r function.zip . -x "*.git*" "node_modules/.cache/*" "deploy.sh" "template.yaml"

echo "☁️  Deploying CloudFormation stack..."
aws cloudformation deploy \
    --template-file template.yaml \
    --stack-name $STACK_NAME \
    --parameter-overrides \
        MongoDBUri="$MONGODB_URI" \
        CronSchedule="cron(15 7 * * ? *)" \
    --capabilities CAPABILITY_IAM \
    --region $REGION

echo "✅ Deployment completed successfully!"

# Get the Lambda function ARN
FUNCTION_ARN=$(aws cloudformation describe-stacks \
    --stack-name $STACK_NAME \
    --region $REGION \
    --query 'Stacks[0].Outputs[?OutputKey==`LambdaFunctionArn`].OutputValue' \
    --output text)

echo "🔗 Lambda Function ARN: $FUNCTION_ARN"
echo "📅 Cron job will run daily at 7:15 AM UTC"
echo "📊 Monitor logs in AWS CloudWatch"

# Clean up
rm -f function.zip

echo "🎉 Setup complete! Your cron job is now running on AWS Lambda."
