"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getYogaTeacherById = exports.getAllYogaTrainers = exports.findYogaTrainerByPhone = exports.createYogaTrainerAccount = void 0;
const db_1 = require("../config/db");
const sequelize_1 = require("sequelize");
const createYogaTrainerAccount = async (fullName, phone, email, passwordHash, qualification, address, state, pincode, expertise, languages, experience, profileImage, bio) => {
    try {
        await db_1.sequelize.query(`CALL CreateYogaTeacher(
        :fullName, :phone, :email, :password, :qualification,
        :address, :state, :pincode, :expertise, :languages,
        :experience, :profileImage, :bio
      )`, {
            replacements: {
                fullName,
                phone,
                email,
                password: passwordHash,
                qualification: qualification || null,
                address: address || null,
                state: state || null,
                pincode: pincode || null,
                expertise: JSON.stringify(expertise),
                languages: JSON.stringify(languages),
                experience: experience || null,
                profileImage: profileImage || null,
                bio: bio || null,
            },
            type: sequelize_1.QueryTypes.RAW,
        });
    }
    catch (error) {
        console.error('Error creating yoga trainer:', error);
        throw error;
    }
};
exports.createYogaTrainerAccount = createYogaTrainerAccount;
// FIND BY PHONE — 100% SAFE
const findYogaTrainerByPhone = async (phone) => {
    try {
        const results = await db_1.sequelize.query(`SELECT * FROM yoga_teachers WHERE phone = :phone LIMIT 1`, {
            replacements: { phone },
            type: sequelize_1.QueryTypes.SELECT,
        });
        if (!results || results.length === 0) {
            return null;
        }
        const row = Array.isArray(results) ? results[0] : results;
        const safeJsonParse = (field) => {
            if (!field)
                return [];
            try {
                return typeof field === 'string' ? JSON.parse(field) : [];
            }
            catch (e) {
                return [];
            }
        };
        return {
            id: row.id,
            fullName: row.fullName,
            phone: row.phone,
            email: row.email,
            password: row.password,
            qualification: row.qualification || '',
            address: row.address || '',
            state: row.state || '',
            pincode: row.pincode || '',
            expertise: safeJsonParse(row.expertise),
            languages: safeJsonParse(row.languages),
            experience: Number(row.experience) || 0,
            profileImage: row.profileImage || '',
            bio: row.bio || '',
            isVerified: Boolean(row.isVerified),
            createdAt: row.createdAt,
            updatedAt: row.updatedAt,
        };
    }
    catch (error) {
        console.error('Error in findYogaTrainerByPhone:', error);
        throw error;
    }
};
exports.findYogaTrainerByPhone = findYogaTrainerByPhone;
// GET ALL TRAINERS — 100% SAFE
const getAllYogaTrainers = async () => {
    try {
        const results = await db_1.sequelize.query(`SELECT * FROM yoga_teachers ORDER BY id DESC`, { type: sequelize_1.QueryTypes.SELECT });
        if (!results || results.length === 0)
            return [];
        const safeJsonParse = (field) => {
            if (!field)
                return [];
            try {
                return typeof field === 'string' ? JSON.parse(field) : [];
            }
            catch {
                return [];
            }
        };
        return results.map((row) => ({
            id: row.id,
            fullName: row.fullName,
            phone: row.phone,
            email: row.email,
            qualification: row.qualification || '',
            address: row.address || '',
            state: row.state || '',
            pincode: row.pincode || '',
            expertise: safeJsonParse(row.expertise),
            languages: safeJsonParse(row.languages),
            experience: Number(row.experience) || 0,
            profileImage: row.profileImage || '',
            bio: row.bio || '',
            isVerified: Boolean(row.isVerified),
            createdAt: row.createdAt,
            updatedAt: row.updatedAt,
        }));
    }
    catch (error) {
        console.error('Error in getAllYogaTrainers:', error);
        throw error;
    }
};
exports.getAllYogaTrainers = getAllYogaTrainers;
// export const getYogaTeacherById = async (id: number): Promise<any | null> => {
//   try {
//     const result: any = await sequelize.query(
//       "CALL GetYogaTeacherById(:id)",
//       {
//         replacements: { id },
//         type: QueryTypes.SELECT,
//       }
//     );
//     if (!result || result.length === 0) return null;
//     const row = result[0];
//     return {
//       fullName: row.fullName || "Unknown Teacher",
//       expertise: row.expertise ? JSON.parse(row.expertise) : [],
//       languages: row.languages ? JSON.parse(row.languages) : [],
//       profileImage: row.profileImage || "/default-avatar.jpg",
//       bio: row.bio || "No bio available",
//     };
//   } catch (error) {
//     console.error("Error fetching yoga teacher:", error);
//     return null;
//   }
// };
const getYogaTeacherById = async (id) => {
    try {
        const result = await db_1.sequelize.query("CALL GetYogaTeacherById(:id)", { replacements: { id } });
        const row = Array.isArray(result) ? result[0] : result;
        if (!row)
            return null;
        return {
            fullName: row.fullName ?? "Unknown Teacher",
            expertise: row.expertise ? JSON.parse(row.expertise) : [],
            languages: row.languages ? JSON.parse(row.languages) : [],
            profileImage: row.profileImage ?? "/default-avatar.jpg",
            bio: row.bio ?? "No bio available",
        };
    }
    catch (error) {
        console.error("Error fetching yoga teacher:", error);
        return null;
    }
};
exports.getYogaTeacherById = getYogaTeacherById;
