"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDateAstrologerWiseAdminEarning = exports.getTodayAstrologerWiseAdminEarning = exports.getAdminEarningByDate = exports.getDateWiseAdminEarning = exports.getTodayAdminEarning = exports.getTotalAdminEarning = exports.fetchAstrologerPayoutHistory = exports.payoutAstrologerAmount = void 0;
const db_1 = require("../config/db");
const sequelize_1 = require("sequelize");
const payoutAstrologerAmount = async ({ astrologer_id, amount, payment_mode, reference_id, remarks, }) => {
    try {
        // 1️⃣ Wallet fetch
        const wallet = await db_1.sequelize.query(`
      SELECT wallet_balance 
      FROM astrologer_wallet 
      WHERE astrologer_id = :id
      `, {
            replacements: { id: astrologer_id },
            type: sequelize_1.QueryTypes.SELECT,
        });
        if (!wallet.length) {
            throw new Error("Wallet not found");
        }
        const beforeBalance = Number(wallet[0].wallet_balance);
        if (amount > beforeBalance) {
            throw new Error("Insufficient wallet balance");
        }
        const afterBalance = beforeBalance - amount;
        // 2️⃣ Update wallet balance
        await db_1.sequelize.query(`
      UPDATE astrologer_wallet
      SET wallet_balance = :after
      WHERE astrologer_id = :id
      `, {
            replacements: { after: afterBalance, id: astrologer_id },
            type: sequelize_1.QueryTypes.UPDATE,
        });
        // 3️⃣ Insert payout history (reference ADMIN se)
        await db_1.sequelize.query(`
      INSERT INTO astrologer_payout_history
      (
        astrologer_id,
        paid_amount,
        before_balance,
        after_balance,
        payment_mode,
        reference_id,
        remarks
      )
      VALUES
      (
        :id,
        :paid,
        :before,
        :after,
        :mode,
        :ref,
        :remarks
      )
      `, {
            replacements: {
                id: astrologer_id,
                paid: amount,
                before: beforeBalance,
                after: afterBalance,
                mode: payment_mode,
                ref: reference_id, // 👈 direct use
                remarks: remarks || null,
            },
            type: sequelize_1.QueryTypes.INSERT,
        });
        return {
            paid_amount: amount,
            remaining_balance: afterBalance,
            reference_id,
        };
    }
    catch (error) {
        console.error("Error in payoutAstrologerAmount:", error);
        throw error;
    }
};
exports.payoutAstrologerAmount = payoutAstrologerAmount;
const fetchAstrologerPayoutHistory = async (astrologer_id) => {
    try {
        const result = await db_1.sequelize.query(`
      SELECT
        p.id,
        p.paid_amount,
        p.before_balance,
        p.after_balance,
        p.payment_mode,
        p.reference_id,
        p.remarks,
        p.created_at,
        a.username AS astrologer_name,
        a.mobileNumber
      FROM astrologer_payout_history p
      JOIN astrologer a ON a.id = p.astrologer_id
      WHERE p.astrologer_id = :id
      ORDER BY p.created_at DESC
      `, {
            replacements: { id: astrologer_id },
            type: sequelize_1.QueryTypes.SELECT,
        });
        return result || [];
    }
    catch (error) {
        console.error("Error fetching payout history:", error);
        throw new Error("Failed to fetch payout history");
    }
};
exports.fetchAstrologerPayoutHistory = fetchAstrologerPayoutHistory;
/**
 * 🔹 Total Admin Earning (All Time)
 */
const getTotalAdminEarning = async () => {
    try {
        const result = await db_1.sequelize.query(`
      SELECT 
        SUM(aum.total_amount * a.commission_percentage / 100)
          AS total_admin_earning
      FROM astrologer_user_messages aum
      JOIN astrologer a 
        ON a.mobileNumber = aum.astrologer_number
      WHERE aum.status = 'end'
      `, { type: sequelize_1.QueryTypes.SELECT });
        return result[0]?.total_admin_earning || 0;
    }
    catch (error) {
        console.error("Error in getTotalAdminEarning:", error);
        throw error;
    }
};
exports.getTotalAdminEarning = getTotalAdminEarning;
/**
 * 🔹 Today Admin Earning
 */
const getTodayAdminEarning = async () => {
    const result = await db_1.sequelize.query(`
    SELECT 
      SUM(aum.total_amount * a.commission_percentage / 100)
        AS today_admin_earning
    FROM astrologer_user_messages aum
    JOIN astrologer a 
      ON a.mobileNumber = aum.astrologer_number
    WHERE aum.status = 'end'
    AND DATE(aum.created_at) = CURDATE()
    `, { type: sequelize_1.QueryTypes.SELECT });
    return result[0]?.today_admin_earning || 0;
};
exports.getTodayAdminEarning = getTodayAdminEarning;
/**
 * 🔹 Date-wise Admin Earning (for charts)
 */
const getDateWiseAdminEarning = async () => {
    return await db_1.sequelize.query(`
    SELECT
      DATE(aum.created_at) AS earning_date,
      SUM(aum.total_amount * a.commission_percentage / 100)
        AS admin_earning
    FROM astrologer_user_messages aum
    JOIN astrologer a 
      ON a.mobileNumber = aum.astrologer_number
    WHERE aum.status = 'end'
    GROUP BY DATE(aum.created_at)
    ORDER BY earning_date DESC
    `, { type: sequelize_1.QueryTypes.SELECT });
};
exports.getDateWiseAdminEarning = getDateWiseAdminEarning;
/**
 * 🔹 Admin Earning by Specific Date
 */
const getAdminEarningByDate = async (date) => {
    try {
        const result = await db_1.sequelize.query(`
      SELECT
        SUM(total_amount * commission_percentage / 100) AS admin_earning
      FROM astrologer_user_messages
      WHERE status = 'end'
      AND DATE(created_at) = :date
      `, {
            replacements: { date },
            type: sequelize_1.QueryTypes.SELECT,
        });
        return result[0]?.admin_earning || 0;
    }
    catch (error) {
        console.error("Error in getAdminEarningByDate:", error);
        throw error;
    }
};
exports.getAdminEarningByDate = getAdminEarningByDate;
const getTodayAstrologerWiseAdminEarning = async () => {
    try {
        const result = await db_1.sequelize.query(`
      SELECT
        a.id AS astrologer_id,
        a.username AS astrologer_name,
        a.mobileNumber,
        COUNT(aum.id) AS total_calls,
        SUM(aum.total_amount) AS total_collection,
        a.commission_percentage,
        SUM(aum.total_amount * a.commission_percentage / 100)
          AS admin_earning
      FROM astrologer_user_messages aum
      JOIN astrologer a 
        ON a.mobileNumber = aum.astrologer_number
      WHERE aum.status = 'end'
      AND DATE(aum.created_at) = CURDATE()
      GROUP BY a.id, DATE(aum.created_at)
      ORDER BY admin_earning DESC
      `, {
            type: sequelize_1.QueryTypes.SELECT,
        });
        return result || [];
    }
    catch (error) {
        console.error("Error in getTodayAstrologerWiseAdminEarning:", error);
        throw error;
    }
};
exports.getTodayAstrologerWiseAdminEarning = getTodayAstrologerWiseAdminEarning;
const getDateAstrologerWiseAdminEarning = async (date) => {
    try {
        const result = await db_1.sequelize.query(`
      SELECT
        a.id AS astrologer_id,
        a.username AS astrologer_name,
        a.mobileNumber,
        COUNT(aum.id) AS total_calls,
        SUM(aum.total_amount) AS total_collection,
        a.commission_percentage,
        SUM(aum.total_amount * a.commission_percentage / 100)
          AS admin_earning
      FROM astrologer_user_messages aum
      JOIN astrologer a 
        ON a.mobileNumber = aum.astrologer_number
      WHERE aum.status = 'end'
      AND DATE(aum.created_at) = :date
      GROUP BY a.id, DATE(aum.created_at)
      ORDER BY admin_earning DESC
      `, {
            replacements: { date },
            type: sequelize_1.QueryTypes.SELECT,
        });
        return result || [];
    }
    catch (error) {
        console.error("Error in getDateAstrologerWiseAdminEarning:", error);
        throw error;
    }
};
exports.getDateAstrologerWiseAdminEarning = getDateAstrologerWiseAdminEarning;
