"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchAllAstrologersWithPriceCommission = exports.toggleAstrologerFreeStatusByMobile = exports.fetchAllAstrologers = exports.getAstrologerPriceAndCommission = exports.updateAstrologerPriceAndCommission = exports.unverifyAstrologer = exports.verifyAstrologer = exports.updateAstrorPass = exports.updateAstrologerPassword = exports.findAstrologerByContact = exports.createAstrologerAccount = void 0;
const db_1 = require("../config/db");
const sequelize_1 = require("sequelize");
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const createAstrologerAccount = async (Astrologer_name, Phone, Email, Address, Pincode, State, Specialization, Languages, Experience, Highest_qualification, Description, Profile_image, passwordHash) => {
    try {
        await db_1.sequelize.query(`CALL CreateAstrologer(
        :Astrologer_name,
        :Phone,
        :Email,
        :Address,
        :Pincode,
        :State,
        :Specialization,
        :Languages,
        :Experience,
        :Highest_qualification,
        :Description,
        :Profile_image,
        :Password
      )`, {
            replacements: {
                Astrologer_name,
                Phone,
                Email,
                Address,
                Pincode,
                State,
                Specialization: JSON.stringify(Specialization),
                Languages: JSON.stringify(Languages),
                Experience,
                Highest_qualification,
                Description,
                Profile_image,
                Password: passwordHash,
            },
            type: sequelize_1.QueryTypes.RAW,
        });
    }
    catch (error) {
        console.error('Error creating astrologer:', error);
        throw error;
    }
};
exports.createAstrologerAccount = createAstrologerAccount;
const findAstrologerByContact = async (contact) => {
    try {
        const result = await db_1.sequelize.query('CALL GetAstrologerWithContact(:contact)', {
            replacements: { contact },
            type: sequelize_1.QueryTypes.SELECT,
        });
        const firstResult = result[0];
        if (!firstResult)
            return null;
        const astrologer = firstResult[Object.keys(firstResult)[0]];
        return astrologer ?? null;
    }
    catch (error) {
        console.error('Error fetching astrologer by contact:', error);
        throw error;
    }
};
exports.findAstrologerByContact = findAstrologerByContact;
const updateAstrologerPassword = async (contact, hashedPassword) => {
    try {
        await db_1.sequelize.query('CALL UpdateAstrologerPassword(:contact, :password)', {
            replacements: { contact, password: hashedPassword },
            type: sequelize_1.QueryTypes.RAW,
        });
    }
    catch (error) {
        console.error('Error updating astrologer password:', error);
        throw error;
    }
};
exports.updateAstrologerPassword = updateAstrologerPassword;
const updateAstrorPass = async (astro_id, current_password, new_password) => {
    try {
        const [astro] = await db_1.sequelize.query('SELECT password FROM astrologer WHERE id = :astro_id', {
            replacements: { astro_id },
            type: sequelize_1.QueryTypes.SELECT,
        });
        if (!astro || !astro.password) {
            throw new Error('Astrologer not found');
        }
        // Step 2: Compare current_password with DB password
        const isMatch = await bcryptjs_1.default.compare(current_password, astro.password);
        if (!isMatch) {
            throw new Error('Incorrect current password');
        }
        // Step 3: Hash new password
        const hashedPassword = await bcryptjs_1.default.hash(new_password, 10);
        // Step 4: Call procedure to update password
        await db_1.sequelize.query('CALL UpdateAstroPassword(:astro_id, :hashedPassword)', {
            replacements: { astro_id, hashedPassword },
            type: sequelize_1.QueryTypes.RAW,
        });
    }
    catch (error) {
        throw error;
    }
};
exports.updateAstrorPass = updateAstrorPass;
const verifyAstrologer = async (astro_id) => {
    try {
        await db_1.sequelize.query("UPDATE astrologer SET is_verified = 1 WHERE id = :astro_id", {
            replacements: { astro_id },
            type: sequelize_1.QueryTypes.UPDATE,
        });
    }
    catch (error) {
        console.error("Error verifying astrologer:", error);
        throw error;
    }
};
exports.verifyAstrologer = verifyAstrologer;
// ✅ Unverify astrologer
const unverifyAstrologer = async (astro_id) => {
    try {
        await db_1.sequelize.query("UPDATE astrologer SET is_verified = 0 WHERE id = :astro_id", {
            replacements: { astro_id },
            type: sequelize_1.QueryTypes.UPDATE,
        });
    }
    catch (error) {
        console.error("Error unverifying astrologer:", error);
        throw error;
    }
};
exports.unverifyAstrologer = unverifyAstrologer;
// 🔹 Update price & commission
const updateAstrologerPriceAndCommission = async (astro_id, price, commission_percentage) => {
    try {
        await db_1.sequelize.query(`UPDATE astrologer 
       SET price = :price, commission_percentage = :commission_percentage 
       WHERE id = :astro_id`, {
            replacements: { astro_id, price, commission_percentage },
            type: sequelize_1.QueryTypes.UPDATE,
        });
    }
    catch (error) {
        console.error("Error updating astrologer price/commission:", error);
        throw error;
    }
};
exports.updateAstrologerPriceAndCommission = updateAstrologerPriceAndCommission;
// 🔹 Get existing price & commission
const getAstrologerPriceAndCommission = async (astro_id) => {
    try {
        const [result] = await db_1.sequelize.query(`SELECT price, commission_percentage 
       FROM astrologer 
       WHERE id = :astro_id`, {
            replacements: { astro_id },
            type: sequelize_1.QueryTypes.SELECT,
        });
        if (!result)
            return null;
        return result;
    }
    catch (error) {
        console.error("Error fetching astrologer price/commission:", error);
        throw error;
    }
};
exports.getAstrologerPriceAndCommission = getAstrologerPriceAndCommission;
const fetchAllAstrologers = async () => {
    try {
        const results = await db_1.sequelize.query(`SELECT 
         id,
         username,
         mobileNumber,
         email,
         address,
         pincode,
         state,
         specialization,
         languages,
         experience,
         education,
         description,
         profileImage,
         password,
         price,
         commission_percentage,
         is_verified
       FROM astrologer`, { type: sequelize_1.QueryTypes.SELECT });
        return results.map((row) => {
            let specialization = [];
            let languages = [];
            try {
                specialization = row.specialization
                    ? JSON.parse(row.specialization)
                    : [];
            }
            catch {
                specialization = row.specialization
                    ? row.specialization.split(",").map((s) => s.trim())
                    : [];
            }
            try {
                languages = row.languages ? JSON.parse(row.languages) : [];
            }
            catch {
                languages = row.languages
                    ? row.languages.split(",").map((l) => l.trim())
                    : [];
            }
            return {
                ...row,
                specialization,
                languages,
            };
        });
    }
    catch (error) {
        console.error("Error fetching astrologers:", error);
        throw error;
    }
};
exports.fetchAllAstrologers = fetchAllAstrologers;
const toggleAstrologerFreeStatusByMobile = async (mobile) => {
    try {
        const result = await db_1.sequelize.query(`CALL ToggleAstrologerFreeStatusByMobile(?)`, {
            replacements: [mobile],
            type: sequelize_1.QueryTypes.RAW,
        });
        let row = null;
        if (result && result[0]) {
            if (Array.isArray(result[0])) {
                row = result[0][0];
            }
            else if (result[0][0]) {
                row = result[0][0];
            }
            else if (typeof result[0] === 'object') {
                row = result[0];
            }
        }
        if (!row && result && result[1]) {
            if (Array.isArray(result[1])) {
                row = result[1][0];
            }
            else if (result[1][0]) {
                row = result[1][0];
            }
        }
        if (!row) {
            throw new Error("No data returned from stored procedure. Check console log.");
        }
        return row;
    }
    catch (error) {
        console.error("Error in toggleAstrologerFreeStatusByMobile:", error);
        throw error;
    }
};
exports.toggleAstrologerFreeStatusByMobile = toggleAstrologerFreeStatusByMobile;
const fetchAllAstrologersWithPriceCommission = async () => {
    try {
        const results = await db_1.sequelize.query(`SELECT 
         id,
         username,
         mobileNumber,
         email,
         address,
         pincode,
         state,
         specialization,
         languages,
         experience,
         education,
         description,
         profileImage,
         price,
         commission_percentage,
         is_verified,
         is_free,
         original_price,
         created_at
       FROM astrologer 
       ORDER BY id DESC`, { type: sequelize_1.QueryTypes.SELECT });
        return results.map((row) => {
            let specialization = [];
            let languages = [];
            try {
                specialization = row.specialization ? JSON.parse(row.specialization) : [];
            }
            catch {
                specialization = row.specialization ? row.specialization.split(",").map((s) => s.trim()) : [];
            }
            try {
                languages = row.languages ? JSON.parse(row.languages) : [];
            }
            catch {
                languages = row.languages ? row.languages.split(",").map((l) => l.trim()) : [];
            }
            return {
                ...row,
                specialization,
                languages,
                price: row.price ?? null,
                commission_percentage: row.commission_percentage ?? null,
                is_free: row.is_free ?? 0,
                original_price: row.original_price ?? null,
            };
        });
    }
    catch (error) {
        console.error("Error fetching astrologers with price/commission:", error);
        throw error;
    }
};
exports.fetchAllAstrologersWithPriceCommission = fetchAllAstrologersWithPriceCommission;
