"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDetails = exports.getAllYogaTrainersCtrl = exports.loginYogaTrainer = exports.registerYogaTrainer = exports.sendOtpYoga = void 0;
const bcryptjs_1 = __importDefault(require("bcryptjs"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const yogaTrainerModel_1 = require("../models/yogaTrainerModel");
const otpStore_1 = require("../otpStore/otpStore");
const axios_1 = __importDefault(require("axios"));
const JWT_SECRET = process.env.JWT_SECRET || 'your-jwt-secret';
async function sendSMS(phone, message) {
    const url = "https://bulksms.smsroot.com/app/smsapi/index.php";
    const params = {
        key: "467DD867AC81DD",
        campaign: "0",
        routeid: "13",
        type: "text",
        contacts: phone,
        senderid: "MOJIJA",
        msg: message,
        template_id: "1707174402396213814"
    };
    try {
        const response = await axios_1.default.get(url, { params });
        console.log("SMS SENT:", response.data);
    }
    catch (error) {
        console.error("SMS FAILED:", error.response?.data || error.message);
    }
}
const sendOtpYoga = async (req, res) => {
    const { phone } = req.body;
    if (!phone || phone.length !== 10) {
        res.status(400).json({ success: false, message: "Valid 10-digit phone required" });
        return;
    }
    const otp = Math.floor(100000 + Math.random() * 900000).toString();
    await (0, otpStore_1.setOtp)(phone, otp);
    console.log(`OTP for Yoga Trainer ${phone}: ${otp}`);
    await sendSMS(phone, `${otp} is your Login OTP. Do not share it with anyone Regards MOJIJA.`);
    res.json({
        success: true,
        message: "OTP sent successfully"
    });
};
exports.sendOtpYoga = sendOtpYoga;
const registerYogaTrainer = async (req, res) => {
    const { fullName, phone, email, password, qualification = '', address = '', state = '', pincode = '', expertise = '[]', languages = '[]', experience = 0, bio = '', otp } = req.body;
    if (!fullName || !phone || !email || !password || !otp) {
        res.status(400).json({ success: false, message: "Required fields missing" });
        return;
    }
    const storedOtp = await (0, otpStore_1.getOtp)(phone);
    if (String(otp).trim() !== String(storedOtp)?.trim()) {
        res.status(400).json({ success: false, message: "Invalid OTP" });
        return;
    }
    try {
        const existing = await (0, yogaTrainerModel_1.findYogaTrainerByPhone)(phone);
        if (existing) {
            res.status(400).json({ success: false, message: "Phone already registered" });
            return;
        }
        const hashedPassword = await bcryptjs_1.default.hash(password, 10);
        const profileImage = req.file ? `https://yourdomain.com/uploads/${req.file.filename}` : '';
        let expertiseArr = [];
        let languagesArr = [];
        try {
            expertiseArr = typeof expertise === 'string' ? JSON.parse(expertise) : [];
        }
        catch {
            expertiseArr = [];
        }
        try {
            languagesArr = typeof languages === 'string' ? JSON.parse(languages) : [];
        }
        catch {
            languagesArr = [];
        }
        await (0, yogaTrainerModel_1.createYogaTrainerAccount)(fullName, phone, email, hashedPassword, qualification, address, state, pincode, expertiseArr, languagesArr, Number(experience) || 0, profileImage, bio);
        await (0, otpStore_1.deleteOtp)(phone);
        res.json({ success: true, message: "Yoga Trainer registered successfully" });
    }
    catch (error) {
        console.error(error);
        res.status(500).json({ success: false, message: error.message || "Server error" });
    }
};
exports.registerYogaTrainer = registerYogaTrainer;
const loginYogaTrainer = async (req, res) => {
    const { phone, password, otp } = req.body;
    if (!phone || !password || !otp) {
        res.status(400).json({ success: false, message: "Phone, password & OTP required" });
        return;
    }
    const storedOtp = await (0, otpStore_1.getOtp)(phone);
    if (String(otp).trim() !== String(storedOtp)?.trim()) {
        res.status(400).json({ success: false, message: "Invalid OTP" });
        return;
    }
    try {
        const trainer = await (0, yogaTrainerModel_1.findYogaTrainerByPhone)(phone);
        if (!trainer) {
            res.status(404).json({ success: false, message: "Trainer not found" });
            return;
        }
        const match = await bcryptjs_1.default.compare(password, trainer.password);
        if (!match) {
            res.status(401).json({ success: false, message: "Invalid password" });
            return;
        }
        const token = jsonwebtoken_1.default.sign({ id: trainer.id, name: trainer.fullName, role: 'yoga_trainer' }, JWT_SECRET, { expiresIn: '7d' });
        await (0, otpStore_1.deleteOtp)(phone);
        const { password: _, ...safeTrainer } = trainer;
        res.json({
            success: true,
            message: "Login successful",
            token,
            trainer: safeTrainer,
        });
    }
    catch (error) {
        res.status(500).json({ success: false, message: "Server error" });
    }
};
exports.loginYogaTrainer = loginYogaTrainer;
const getAllYogaTrainersCtrl = async (req, res) => {
    try {
        const trainers = await (0, yogaTrainerModel_1.getAllYogaTrainers)();
        res.json({ success: true, count: trainers.length, data: trainers });
    }
    catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
};
exports.getAllYogaTrainersCtrl = getAllYogaTrainersCtrl;
// export const getDetails = async (req: Request, res: Response) => {
//   try {
//     const { id } = req.params;
//     const trainer = await getYogaTeacherById(Number(id));
//     if (!trainer) {
//       return res.status(404).json({ success: false, message: "Trainer not found" });
//     }
//     res.json({ success: true, data: trainer });
//   } catch (error) {
//     res.status(500).json({ success: false, message: "Server error" });
//   }
// };
const getDetails = async (req, res) => {
    try {
        const { id } = req.params;
        const trainer = await (0, yogaTrainerModel_1.getYogaTeacherById)(Number(id));
        if (!trainer) {
            res.status(404).json({ success: false, message: "Trainer not found" });
            return;
        }
        res.json({ success: true, data: trainer });
    }
    catch (error) {
        console.error("Error in getDetails:", error);
        res.status(500).json({ success: false, message: "Server error" });
    }
};
exports.getDetails = getDetails;
