import { Request, Response } from 'express';
import bcrypt from 'bcryptjs';
import jwt from 'jsonwebtoken';
import { createYogaTrainerAccount, findYogaTrainerByPhone, getAllYogaTrainers, getYogaTeacherById } from '../models/yogaTrainerModel';
import { getOtp, deleteOtp, setOtp } from '../otpStore/otpStore';
import axios from 'axios';

const JWT_SECRET = process.env.JWT_SECRET || 'your-jwt-secret';

interface MulterRequest extends Request {
  file?: Express.Multer.File;
}
async function sendSMS(phone: string, message: string): Promise<void> {

  const url = "https://bulksms.smsroot.com/app/smsapi/index.php";

  const params = {
    key: "467DD867AC81DD",
    campaign: "0",
    routeid: "13",
    type: "text",
    contacts: phone,
    senderid: "MOJIJA",
    msg: message,
    template_id: "1707174402396213814"
  };

  try {
    const response = await axios.get(url, { params });
    console.log("SMS SENT:", response.data);
  } catch (error: any) {
    console.error("SMS FAILED:", error.response?.data || error.message);
  }
}


export const sendOtpYoga = async (req: Request, res: Response): Promise<void> => {
  const { phone } = req.body;

  if (!phone || phone.length !== 10) {
    res.status(400).json({ success: false, message: "Valid 10-digit phone required" });
    return;
  }

  const otp = Math.floor(100000 + Math.random() * 900000).toString();
  await setOtp(phone, otp);

  console.log(`OTP for Yoga Trainer ${phone}: ${otp}`); 

await sendSMS(phone, `${otp} is your Login OTP. Do not share it with anyone Regards MOJIJA.`);


  res.json({ 
    success: true, 
    message: "OTP sent successfully" 
  });
};

export const registerYogaTrainer = async (req: MulterRequest, res: Response): Promise<void> => {
  const {
    fullName, phone, email, password, qualification = '',
    address = '', state = '', pincode = '', expertise = '[]',
    languages = '[]', experience = 0, bio = '', otp
  } = req.body;

  if (!fullName || !phone || !email || !password || !otp) {
    res.status(400).json({ success: false, message: "Required fields missing" });
    return;
  }

  const storedOtp = await getOtp(phone);
  if (String(otp).trim() !== String(storedOtp)?.trim()) {
    res.status(400).json({ success: false, message: "Invalid OTP" });
    return;
  }

  try {
    const existing = await findYogaTrainerByPhone(phone);
    if (existing) {
      res.status(400).json({ success: false, message: "Phone already registered" });
      return;
    }

    const hashedPassword = await bcrypt.hash(password, 10);
    const profileImage = req.file ? `https://yourdomain.com/uploads/${req.file.filename}` : '';

    let expertiseArr: string[] = [];
    let languagesArr: string[] = [];

    try {
      expertiseArr = typeof expertise === 'string' ? JSON.parse(expertise) : [];
    } catch {
      expertiseArr = [];
    }
    try {
      languagesArr = typeof languages === 'string' ? JSON.parse(languages) : [];
    } catch {
      languagesArr = [];
    }

    await createYogaTrainerAccount(
      fullName,
      phone,
      email,
      hashedPassword,
      qualification,
      address,
      state,
      pincode,
      expertiseArr,
      languagesArr,
      Number(experience) || 0,
      profileImage,
      bio
    );

    await deleteOtp(phone);

    res.json({ success: true, message: "Yoga Trainer registered successfully" });

  } catch (error: any) {
    console.error(error);
    res.status(500).json({ success: false, message: error.message || "Server error" });
  }
};

export const loginYogaTrainer = async (req: Request, res: Response): Promise<void> => {
  const { phone, password, otp } = req.body;

  if (!phone || !password || !otp) {
    res.status(400).json({ success: false, message: "Phone, password & OTP required" });
    return;
  }

  const storedOtp = await getOtp(phone);
  if (String(otp).trim() !== String(storedOtp)?.trim()) {
    res.status(400).json({ success: false, message: "Invalid OTP" });
    return;
  }

  try {
    const trainer = await findYogaTrainerByPhone(phone);
    if (!trainer) {
      res.status(404).json({ success: false, message: "Trainer not found" });
      return;
    }

    const match = await bcrypt.compare(password, trainer.password);
    if (!match) {
      res.status(401).json({ success: false, message: "Invalid password" });
      return;
    }

    const token = jwt.sign(
      { id: trainer.id, name: trainer.fullName, role: 'yoga_trainer' },
      JWT_SECRET,
      { expiresIn: '7d' }
    );

    await deleteOtp(phone);

    const { password: _, ...safeTrainer } = trainer;

    res.json({
      success: true,
      message: "Login successful",
      token,
      trainer: safeTrainer,
    });

  } catch (error) {
    res.status(500).json({ success: false, message: "Server error" });
  }
};

export const getAllYogaTrainersCtrl = async (req: Request, res: Response) => {
  try {
    const trainers = await getAllYogaTrainers();
    res.json({ success: true, count: trainers.length, data: trainers });
  } catch (error: any) {
    res.status(500).json({ success: false, message: error.message });
  }
};

// export const getDetails = async (req: Request, res: Response) => {
//   try {
//     const { id } = req.params;
//     const trainer = await getYogaTeacherById(Number(id));

//     if (!trainer) {
//       return res.status(404).json({ success: false, message: "Trainer not found" });
//     }

//     res.json({ success: true, data: trainer });
//   } catch (error) {
//     res.status(500).json({ success: false, message: "Server error" });
//   }
// };

export const getDetails = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const trainer = await getYogaTeacherById(Number(id));

    if (!trainer) {
      res.status(404).json({ success: false, message: "Trainer not found" });
      return;
    }

    res.json({ success: true, data: trainer });
  } catch (error) {
    console.error("Error in getDetails:", error);
    res.status(500).json({ success: false, message: "Server error" });
  }
};