import { Request, Response } from "express";
import { payoutAstrologerAmount, fetchAstrologerPayoutHistory } from "../models/payoutModel";

export const payoutAstrologer = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const {
      astrologer_id,
      amount,
      payment_mode,
      reference_id,
      remarks,
    } = req.body;

    if (!astrologer_id || !amount || !payment_mode || !reference_id) {
      res.status(400).json({
        success: false,
        message: "Missing required fields",
      });
      return;
    }

    const data = await payoutAstrologerAmount({
      astrologer_id: Number(astrologer_id),
      amount: Number(amount),
      payment_mode,
      reference_id,
      remarks,
    });

    res.status(200).json({
      success: true,
      message: "Payout recorded successfully",
      data,
    });
  } catch (error: any) {
    res.status(400).json({
      success: false,
      message: error.message || "Failed to payout astrologer",
    });
  }
};



export const getAstrologerPayoutHistory = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const { astrologer_id } = req.params;

    if (!astrologer_id) {
      res.status(400).json({
        success: false,
        message: "Astrologer ID is required",
      });
      return;
    }

    const data = await fetchAstrologerPayoutHistory(
      Number(astrologer_id)
    );

    res.status(200).json({
      success: true,
      count: data.length,
      data,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message: error.message || "Failed to fetch payout history",
    });
  }
};


import {
  getTotalAdminEarning,
  getTodayAdminEarning,
  getDateWiseAdminEarning,
  getAdminEarningByDate,
  getTodayAstrologerWiseAdminEarning,
  getDateAstrologerWiseAdminEarning
} from "../models/payoutModel";

/**
 * 🔹 Total Admin Earning
 */
export const totalAdminEarning = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const amount = await getTotalAdminEarning();

    res.status(200).json({
      success: true,
      total_admin_earning: amount,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message: error.message || "Failed to fetch total admin earning",
    });
  }
};

/**
 * 🔹 Today Admin Earning
 */
export const todayAdminEarning = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const amount = await getTodayAdminEarning();

    res.status(200).json({
      success: true,
      today_admin_earning: amount,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message: error.message || "Failed to fetch today admin earning",
    });
  }
};

/**
 * 🔹 Date-wise Admin Earning
 */
export const dateWiseAdminEarning = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const data = await getDateWiseAdminEarning();

    res.status(200).json({
      success: true,
      count: data.length,
      data,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message: error.message || "Failed to fetch date-wise earning",
    });
  }
};

/**
 * 🔹 Admin Earning by Date
 */
export const adminEarningByDate = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const { date } = req.params;

    if (!date) {
      res.status(400).json({
        success: false,
        message: "Date is required",
      });
      return;
    }

    const amount = await getAdminEarningByDate(date);

    res.status(200).json({
      success: true,
      date,
      admin_earning: amount,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message: error.message || "Failed to fetch admin earning by date",
    });
  }
};





export const todayAstrologerWiseAdminEarning = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const data = await getTodayAstrologerWiseAdminEarning();

    res.status(200).json({
      success: true,
      date: new Date().toISOString().split("T")[0],
      count: data.length,
      data,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message: error.message || "Failed to fetch astrologer-wise admin earning",
    });
  }
};



export const dateAstrologerWiseAdminEarning = async (
  req: Request,
  res: Response
): Promise<void> => {
  try {
    const { date } = req.params;

    if (!date) {
      res.status(400).json({
        success: false,
        message: "Date is required (YYYY-MM-DD)",
      });
      return;
    }

    const data = await getDateAstrologerWiseAdminEarning(date);

    res.status(200).json({
      success: true,
      date,
      count: data.length,
      data,
    });
  } catch (error: any) {
    res.status(500).json({
      success: false,
      message:
        error.message ||
        "Failed to fetch date-wise astrologer admin earning",
    });
  }
};