import { Request, Response } from "express";
import bcrypt from "bcryptjs";
import {
  createPanditAccount,
  getAllPandits,
  getPanditById,
  findPanditByContact,
  verifyPanditById,
  unverifyPanditById,
} from "../models/panditModel";

export const registerPandit = async (req: Request, res: Response): Promise<void> => {
  try {
    const {
      username,
      mobileNumber,
      email,
      address,
      pincode,
      state,
      PoojaSpecialist,
      languages,
      experience,
      education,
      description,
      gender,
    } = req.body;

    const files = req.files as { [fieldname: string]: Express.Multer.File[] };

    //   : '';
    const profileImage = req.file
      ? `https://backend.navambhaw.com/uploads/${req.file.filename}`
      : '';

    const passwordHash = await bcrypt.hash(req.body.password, 10);

    await createPanditAccount(
      username,
      email,
      passwordHash,
      address,
      gender,
      mobileNumber,
      pincode,
      state,
      PoojaSpecialist,
      languages,
      experience,
      education,
      description,
      profileImage || ""
    );

    res.status(201).json({
      success: true,
      message: "Pandit registered successfully",
    });
  } catch (error: any) {
    console.error("Error registering pandit:", error);
    res.status(500).json({ success: false, message: error.message });
  }
};

// Baaki sab functions same rahenge
export const fetchAllPandits = async (req: Request, res: Response): Promise<void> => {
  try {
    const pandits = await getAllPandits();
    res.status(200).json({ success: true, data: pandits });
  } catch (error: any) {
    console.error("Error fetching all pandits:", error);
    res.status(500).json({ success: false, message: error.message });
  }
};

export const fetchPanditById = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const pandit = await getPanditById(Number(id));
    if (!pandit) {
      res.status(404).json({ success: false, message: "Pandit not found" });
      return;
    }
    res.status(200).json({ success: true, data: pandit });
  } catch (error: any) {
    console.error("Error fetching pandit by id:", error);
    res.status(500).json({ success: false, message: error.message });
  }
};

export const fetchPanditByPhone = async (req: Request, res: Response): Promise<void> => {
  try {
    const { phone } = req.params;
    const pandit = await findPanditByContact(phone);
    if (!pandit) {
      res.status(404).json({ success: false, message: "Pandit not found" });
      return;
    }
    res.status(200).json({ success: true, data: pandit });
  } catch (error: any) {
    console.error("Error fetching pandit by phone:", error);
    res.status(500).json({ success: false, message: error.message });
  }
};

export const verifyPandit = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const result = await verifyPanditById(Number(id));
    if (result.affectedRows === 0) {
      res.status(404).json({ success: false, message: "Pandit not found" });
      return;
    }
    res.status(200).json({ success: true, message: "Pandit verified successfully" });
  } catch (error: any) {
    console.error("Error verifying pandit:", error);
    res.status(500).json({ success: false, message: error.message });
  }
};

export const unverifyPandit = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id } = req.params;
    const result = await unverifyPanditById(Number(id));
    if (result.affectedRows === 0) {
      res.status(404).json({ success: false, message: "Pandit not found" });
      return;
    }
    res.status(200).json({ success: true, message: "Pandit unverified successfully" });
  } catch (error: any) {
    console.error("Error unverifying pandit:", error);
    res.status(500).json({ success: false, message: error.message });
  }
};