import { Request, Response } from 'express';
import {
  addAstrologerFreeSlot,
  getAstrologerFreeSlots,
  updateAstrologerFreeSlot,
  deleteAstrologerFreeSlot
} from '../models/astrologerFreeSlotModel';


export const addFreeSlot = async (req: Request, res: Response): Promise<void> => {
  try {
    const { astrologer_id, start_time, end_time, repeat_daily } = req.body;

    if (!astrologer_id || !start_time || !end_time) {
      res.status(400).json({ success: false, message: 'Required fields missing' });
      return;
    }

    await addAstrologerFreeSlot(
      astrologer_id,
      start_time,
      end_time,
      repeat_daily ? 1 : 0
    );

    res.status(200).json({
      success: true,
      message: 'Free slot added successfully'
    });

  } catch (error) {
    console.error(error);
    res.status(500).json({ success: false, message: 'Failed to add slot' });
  }
};

export const getFreeSlots = async (req: Request, res: Response): Promise<void> => {
  try {
    const astrologer_id = Number(req.query.astrologer_id);

    if (!astrologer_id) {
      res.status(400).json({ success: false, message: 'astrologer_id required' });
      return;
    }

    const data = await getAstrologerFreeSlots(astrologer_id);

    res.status(200).json({ success: true, data });

  } catch (error) {
    res.status(500).json({ success: false, message: 'Failed to fetch slots' });
  }
};

export const updateFreeSlot = async (req: Request, res: Response): Promise<void> => {
  try {
    const { id, start_time, end_time, repeat_daily } = req.body;

    await updateAstrologerFreeSlot(
      id,
      start_time,
      end_time,
      repeat_daily ? 1 : 0
    );

    res.status(200).json({
      success: true,
      message: 'Slot updated successfully'
    });

  } catch (error) {
    res.status(500).json({ success: false, message: 'Failed to update slot' });
  }
};

export const deleteFreeSlot = async (req: Request, res: Response): Promise<void> => {
  try {
    const id = Number(req.query.id);

    await deleteAstrologerFreeSlot(id);

    res.status(200).json({
      success: true,
      message: 'Slot deleted successfully'
    });

  } catch (error) {
    res.status(500).json({ success: false, message: 'Failed to delete slot' });
  }
};
